#!/usr/bin/env node

/**
 * @skematica/secop-bridge
 * Universal SSE Bridge for SECOP MCP Server.
 * 
 * This bridge connects a local Stdio MCP client (like Claude Desktop or VS Code)
 * to the remote SECOP SSE Endpoint (https://secop.skemati.ca/sse).
 * 
 * Usage:
 * npx @skematica/secop-bridge
 * (Requires MCP_API_KEY environment variable)
 */

const https = require('https');
const readline = require('readline');
const fs = require('fs');
const path = require('path');
const os = require('os');

// --- Configuration ---
const DEFAULT_URL = 'https://secop.skemati.ca/sse';
const SSE_URL = 'https://secop.skemati.ca/sse';
const API_KEY = 'YOUR_KEY';
const DEBUG = process.env.MCP_DEBUG === 'true';

// --- Logging System ---
const LOG_FILE = path.join(os.tmpdir(), 'secop_mcp_bridge.log');

function log(msg) {
    if (!DEBUG) return;
    try {
        const timestamp = new Date().toISOString();
        fs.appendFileSync(LOG_FILE, `[${timestamp}] ${msg}\n`);
    } catch (e) { }
}

function error(msg) {
    try {
        const timestamp = new Date().toISOString();
        fs.appendFileSync(LOG_FILE, `[${timestamp}] [ERROR] ${msg}\n`);
    } catch (e) { }
    console.error(`[SecopBridge] ${msg}`);
}

// --- Validation ---
log('--- Bridge Starting ---');
log(`Target: ${SSE_URL}`);

if (!API_KEY) {
    error('CRITICAL: MCP_API_KEY environment variable is missing.');
    process.exit(1);
}

// --- State ---
let sessionId = null;
let pendingMessages = [];
let sseRequest = null;
let sseBuffer = '';
let isReconnecting = false;
let messageQueueLocked = false;

const MESSAGES_URL = SSE_URL.replace('/sse', '/messages');

// --- 1. SSE Connection Management ---
function connectSSE() {
    if (sseRequest) {
        try { sseRequest.destroy(); } catch (e) { }
    }

    log(`Connecting to SSE...`);

    const options = {
        headers: {
            'Authorization': `Bearer ${API_KEY}`,
            'Cache-Control': 'no-cache',
            'Accept': 'text/event-stream',
            'User-Agent': 'Skematica-Bridge/1.0.1'
        }
    };

    sseRequest = https.get(SSE_URL, options, (res) => {
        log(`SSE Connection Status: ${res.statusCode}`);

        if (res.statusCode === 401 || res.statusCode === 403) {
            error(`Authentication Failed (${res.statusCode}). Please check your MCP_API_KEY.`);
            process.exit(1);
        }

        if (res.statusCode >= 400) {
            log(`Server returned error: ${res.statusCode}. Retrying in 5s...`);
            setTimeout(connectSSE, 5000);
            return;
        }

        res.on('data', (chunk) => {
            sseBuffer += chunk.toString();
            let eventEnd;
            while ((eventEnd = sseBuffer.indexOf('\n\n')) !== -1) {
                const eventBlock = sseBuffer.slice(0, eventEnd);
                sseBuffer = sseBuffer.slice(eventEnd + 2);
                processEvent(eventBlock);
            }
        });

        res.on('end', () => {
            log('SSE Connection closed by server. Reconnecting...');
            sessionId = null;
            setTimeout(connectSSE, 1000);
        });
    });

    sseRequest.on('error', (e) => {
        log(`Connection Error: ${e.message}. Retrying in 5s...`);
        sessionId = null;
        setTimeout(connectSSE, 5000);
    });
}

function processEvent(eventBlock) {
    const lines = eventBlock.split('\n');
    let eventType = 'message';

    for (const line of lines) {
        if (line.startsWith('event: ')) {
            eventType = line.slice(7).trim();
        } else if (line.startsWith('data: ')) {
            const data = line.slice(6).trim();
            handleData(data, eventType);
        }
    }
}

function handleData(data, eventType) {
    // 1. Session Initialization
    if (data.includes('?sessionId=')) {
        const match = data.match(/sessionId=([^&]*)/);
        if (match) {
            sessionId = match[1];
            log(`Session Established: ${sessionId}`);
            messageQueueLocked = false;
            flushPendingMessages();
        }
        return;
    }

    // 2. JSON-RPC Message Forwarding
    try {
        const json = JSON.parse(data);
        log(`Forwarding to client: ${json.method || json.id}`);
        process.stdout.write(JSON.stringify(json) + '\n');
    } catch (e) {
        if (data.length > 0 && data !== ':') {
            log(`Non-JSON data ignored.`);
        }
    }
}

// --- 2. Input Handling (Client -> Server) ---
const rl = readline.createInterface({ input: process.stdin, terminal: false });

rl.on('line', (line) => {
    try {
        const message = JSON.parse(line);
        // Always queue first, then flush if ready
        pendingMessages.push(message);

        if (sessionId && !messageQueueLocked) {
            flushPendingMessages();
        } else {
            log('Queuing message (session not ready)');
        }
    } catch (e) {
        log(`Invalid Input from Client: ${line}`);
    }
});

function flushPendingMessages() {
    if (messageQueueLocked || !sessionId) return;

    log(`Flushing ${pendingMessages.length} pending messages`);

    // Process queue
    while (pendingMessages.length > 0) {
        const msg = pendingMessages.shift();
        sendMessage(msg);
    }
}

function sendMessage(message) {
    const postData = JSON.stringify(message);
    const postUrl = new URL(MESSAGES_URL);
    postUrl.searchParams.append('sessionId', sessionId);

    const postReq = https.request(postUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${API_KEY}`,
            'Content-Length': Buffer.byteLength(postData),
            'User-Agent': 'Skematica-Bridge/1.0.1'
        }
    }, (res) => {
        if (res.statusCode >= 400) {
            log(`POST Error ${res.statusCode} for method ${message.method}`);

            if (res.statusCode === 404) {
                // Session Expired logic
                log('Session expired (404). Triggering reconnection...');
                sessionId = null;
                messageQueueLocked = true;

                // Re-queue this message at the front
                pendingMessages.unshift(message);

                // Trigger reconnection immediately
                connectSSE();
            }
        }
    });

    postReq.on('error', (e) => {
        log(`POST Request Error: ${e.message}`);
        // Network error? Re-queue might be dangerous if unlimited, 
        // but for now we log. If it was a network drop, SSE error handler should trigger reconnect.
    });

    postReq.write(postData);
    postReq.end();
}

// Start
connectSSE();
